// Terminal Chat - Content Script
(function() {
  'use strict';

  // ============================================
  // CONFIGURATION - UPDATE THIS AFTER DEPLOYING
  // ============================================
  const CONFIG = {
    API_URL: 'https://terminalchat-nine.vercel.app/api/messages',
    PUSHER_KEY: '26d45daced8b2f091b88',
    PUSHER_CLUSTER: 'mt1',
  };

  // ============================================
  // INJECT PAGE SCRIPT FOR PHANTOM ACCESS
  // ============================================
  function injectScript() {
    const script = document.createElement('script');
    script.src = chrome.runtime.getURL('injected.js');
    script.onload = function() {
      this.remove();
    };
    (document.head || document.documentElement).appendChild(script);
  }
  injectScript();

  // Message handling for injected script communication
  const pendingRequests = new Map();
  let requestId = 0;

  window.addEventListener('message', (event) => {
    if (event.source !== window) return;
    if (!event.data || event.data.source !== 'terminal-chat-injected') return;

    const { id } = event.data;
    if (id && pendingRequests.has(id)) {
      const { resolve, reject } = pendingRequests.get(id);
      pendingRequests.delete(id);

      if (event.data.error) {
        reject(event.data);
      } else {
        resolve(event.data);
      }
    }
  });

  function sendToPage(type, data = {}) {
    return new Promise((resolve, reject) => {
      const id = ++requestId;
      pendingRequests.set(id, { resolve, reject });

      window.postMessage({
        source: 'terminal-chat-content',
        type,
        id,
        ...data
      }, '*');

      // Timeout after 30 seconds
      setTimeout(() => {
        if (pendingRequests.has(id)) {
          pendingRequests.delete(id);
          reject({ error: 'TIMEOUT' });
        }
      }, 30000);
    });
  }

  // ============================================
  // COLOR GENERATOR - Consistent colors per wallet
  // ============================================
  const walletColors = {};
  const colorPalette = [
    '#FF6B6B', '#4ECDC4', '#45B7D1', '#96CEB4', '#FFEAA7',
    '#DDA0DD', '#98D8C8', '#F7DC6F', '#BB8FCE', '#85C1E9',
    '#F8B500', '#00CED1', '#FF69B4', '#32CD32', '#FFD700',
    '#FF7F50', '#87CEEB', '#DA70D6', '#00FA9A', '#FF6347'
  ];

  function getWalletColor(wallet) {
    if (!walletColors[wallet]) {
      // Generate consistent color based on wallet address
      let hash = 0;
      for (let i = 0; i < wallet.length; i++) {
        hash = wallet.charCodeAt(i) + ((hash << 5) - hash);
      }
      walletColors[wallet] = colorPalette[Math.abs(hash) % colorPalette.length];
    }
    return walletColors[wallet];
  }

  function shortenWallet(wallet) {
    return wallet.slice(0, 4) + '...' + wallet.slice(-4);
  }

  // ============================================
  // STATE
  // ============================================
  let state = {
    wallet: null,
    connected: false,
    minimized: false,
    hidden: false,
    pusher: null,
    channel: null,
    lastMessageTime: 0,
    cooldownInterval: null,
  };

  const COOLDOWN_MS = 5000; // 5 second cooldown

  // ============================================
  // CREATE WIDGET HTML
  // ============================================
  function createWidget() {
    // Toggle button (shown when widget is closed)
    const toggleBtn = document.createElement('button');
    toggleBtn.id = 'tc-toggle-btn';
    toggleBtn.className = 'hidden';
    toggleBtn.innerHTML = `
      <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
        <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"/>
      </svg>
    `;
    document.body.appendChild(toggleBtn);

    // Main widget
    const widget = document.createElement('div');
    widget.id = 'terminal-chat-widget';
    widget.innerHTML = `
      <div id="tc-titlebar">
        <span id="tc-title">⌘ Terminal Chat</span>
        <div id="tc-controls">
          <button class="tc-btn minimize" id="tc-minimize">_</button>
          <button class="tc-btn close" id="tc-close">×</button>
        </div>
      </div>
      
      <div id="tc-login">
        <div id="tc-login-prompt">
          > Connect your Phantom wallet to chat<br>
          > Cross-platform chat for degens_
        </div>
        <button id="tc-connect-wallet">🔮 Connect Phantom</button>
        <div id="tc-login-error"></div>
      </div>
      
      <div id="tc-chat-container" style="display: none;">
        <div id="tc-messages"></div>
        <div id="tc-input-area">
          <span id="tc-input-prefix">&gt;</span>
          <input type="text" id="tc-input" placeholder="Type a message..." maxlength="500" autocomplete="off">
        </div>
      </div>
      
      <div id="tc-statusbar">
        <div id="tc-connection-status">
          <span id="tc-connection-dot"></span>
          <span id="tc-connection-text">Disconnected</span>
        </div>
        <div id="tc-wallet-display"></div>
      </div>
    `;
    document.body.appendChild(widget);

    // Make widget draggable
    makeDraggable(widget, document.getElementById('tc-titlebar'));

    // Set up event listeners
    setupEventListeners();
  }

  // ============================================
  // DRAGGABLE FUNCTIONALITY
  // ============================================
  function makeDraggable(element, handle) {
    let offsetX, offsetY, isDragging = false;

    handle.addEventListener('mousedown', (e) => {
      if (e.target.tagName === 'BUTTON') return;
      isDragging = true;
      offsetX = e.clientX - element.getBoundingClientRect().left;
      offsetY = e.clientY - element.getBoundingClientRect().top;
      element.style.transition = 'none';
    });

    document.addEventListener('mousemove', (e) => {
      if (!isDragging) return;
      const x = e.clientX - offsetX;
      const y = e.clientY - offsetY;
      
      // Keep within viewport
      const maxX = window.innerWidth - element.offsetWidth;
      const maxY = window.innerHeight - element.offsetHeight;
      
      element.style.left = Math.max(0, Math.min(x, maxX)) + 'px';
      element.style.top = Math.max(0, Math.min(y, maxY)) + 'px';
      element.style.right = 'auto';
      element.style.bottom = 'auto';
    });

    document.addEventListener('mouseup', () => {
      isDragging = false;
      element.style.transition = '';
    });
  }

  // ============================================
  // EVENT LISTENERS
  // ============================================
  function setupEventListeners() {
    // Connect wallet button
    document.getElementById('tc-connect-wallet').addEventListener('click', connectWallet);

    // Minimize button
    document.getElementById('tc-minimize').addEventListener('click', () => {
      const widget = document.getElementById('terminal-chat-widget');
      state.minimized = !state.minimized;
      widget.classList.toggle('minimized', state.minimized);
      document.getElementById('tc-minimize').textContent = state.minimized ? '□' : '_';
    });

    // Close button
    document.getElementById('tc-close').addEventListener('click', () => {
      document.getElementById('terminal-chat-widget').classList.add('hidden');
      document.getElementById('tc-toggle-btn').classList.remove('hidden');
    });

    // Toggle button (reopen widget)
    document.getElementById('tc-toggle-btn').addEventListener('click', () => {
      document.getElementById('terminal-chat-widget').classList.remove('hidden');
      document.getElementById('tc-toggle-btn').classList.add('hidden');
    });

    // Message input
    document.getElementById('tc-input').addEventListener('keypress', (e) => {
      if (e.key === 'Enter' && !e.shiftKey) {
        e.preventDefault();
        sendMessage();
      }
    });
  }

  // ============================================
  // PHANTOM WALLET CONNECTION
  // ============================================
  async function connectWallet() {
    const errorEl = document.getElementById('tc-login-error');
    const connectBtn = document.getElementById('tc-connect-wallet');
    errorEl.textContent = '';
    connectBtn.disabled = true;
    connectBtn.textContent = '🔮 Connecting...';

    try {
      // Use injected script to connect to Phantom
      const response = await sendToPage('CONNECT_PHANTOM');

      if (response.wallet) {
        state.wallet = response.wallet;
        
        // Update UI
        showChat();
        
        // Connect to Pusher and load messages
        connectPusher();
        loadMessages();
      }
      
    } catch (err) {
      console.error('Wallet connection error:', err);
      
      if (err.error === 'PHANTOM_NOT_FOUND') {
        errorEl.textContent = '> Phantom wallet not found. Please install it.';
        setTimeout(() => {
          window.open('https://phantom.app/', '_blank');
        }, 1000);
      } else if (err.error === 'USER_REJECTED') {
        errorEl.textContent = '> Connection rejected by user';
      } else if (err.error === 'TIMEOUT') {
        errorEl.textContent = '> Connection timed out. Try again.';
      } else {
        errorEl.textContent = '> Error: ' + (err.error || 'Failed to connect');
      }
    } finally {
      connectBtn.disabled = false;
      connectBtn.textContent = '🔮 Connect Phantom';
    }
  }

  // ============================================
  // UI STATE CHANGES
  // ============================================
  function showChat() {
    document.getElementById('tc-login').style.display = 'none';
    document.getElementById('tc-chat-container').style.display = 'flex';
    document.getElementById('tc-wallet-display').textContent = shortenWallet(state.wallet);
    document.getElementById('tc-input').focus();
  }

  function updateConnectionStatus(connected) {
    state.connected = connected;
    const dot = document.getElementById('tc-connection-dot');
    const text = document.getElementById('tc-connection-text');
    
    dot.classList.toggle('connected', connected);
    text.textContent = connected ? 'Connected' : 'Disconnected';
  }

  // ============================================
  // PUSHER REAL-TIME CONNECTION
  // ============================================
  function connectPusher() {
    if (typeof Pusher === 'undefined') {
      console.error('Pusher not loaded');
      addSystemMessage('Error: Real-time connection failed');
      return;
    }

    state.pusher = new Pusher(CONFIG.PUSHER_KEY, {
      cluster: CONFIG.PUSHER_CLUSTER,
    });

    state.channel = state.pusher.subscribe('chat');
    
    state.channel.bind('new-message', (message) => {
      addMessage(message);
    });

    state.pusher.connection.bind('connected', () => {
      updateConnectionStatus(true);
    });

    state.pusher.connection.bind('disconnected', () => {
      updateConnectionStatus(false);
    });

    state.pusher.connection.bind('error', (err) => {
      console.error('Pusher error:', err);
      updateConnectionStatus(false);
    });
  }

  // ============================================
  // MESSAGE HANDLING
  // ============================================
  async function loadMessages() {
    try {
      addSystemMessage('Loading messages...');
      
      const response = await fetch(CONFIG.API_URL);
      const data = await response.json();
      
      // Clear loading message
      const messagesEl = document.getElementById('tc-messages');
      messagesEl.innerHTML = '';
      
      if (data.messages && data.messages.length > 0) {
        data.messages.forEach(msg => {
          if (typeof msg === 'string') {
            addMessage(JSON.parse(msg));
          } else {
            addMessage(msg);
          }
        });
      } else {
        addSystemMessage('No messages yet. Be the first to chat!');
      }
      
    } catch (err) {
      console.error('Failed to load messages:', err);
      addSystemMessage('Error loading messages. Check console.');
    }
  }

  // ============================================
  // COOLDOWN HANDLING
  // ============================================
  function updateCooldown() {
    const now = Date.now();
    const remaining = Math.ceil((COOLDOWN_MS - (now - state.lastMessageTime)) / 1000);
    const input = document.getElementById('tc-input');
    
    if (remaining > 0) {
      input.placeholder = `Wait ${remaining}s...`;
      input.disabled = true;
    } else {
      input.placeholder = 'Type a message...';
      input.disabled = false;
      if (state.cooldownInterval) {
        clearInterval(state.cooldownInterval);
        state.cooldownInterval = null;
      }
    }
  }

  async function sendMessage() {
    const input = document.getElementById('tc-input');
    const text = input.value.trim();
    
    if (!text || !state.wallet) return;

    // Check cooldown
    const now = Date.now();
    if (now - state.lastMessageTime < COOLDOWN_MS) {
      updateCooldown();
      return;
    }
    
    input.value = '';
    input.disabled = true;
    state.lastMessageTime = now;

    // Start cooldown
    state.cooldownInterval = setInterval(updateCooldown, 100);
    updateCooldown();

    try {
      const response = await fetch(CONFIG.API_URL, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          wallet: state.wallet,
          text: text,
        }),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to send message');
      }
      
    } catch (err) {
      console.error('Failed to send message:', err);
      addSystemMessage('Error: ' + err.message);
      // Restore the message so user can retry
      input.value = text;
      // Reset cooldown on error
      state.lastMessageTime = 0;
      if (state.cooldownInterval) {
        clearInterval(state.cooldownInterval);
        state.cooldownInterval = null;
      }
      input.disabled = false;
      input.placeholder = 'Type a message...';
    }
  }

  function addMessage(message) {
    const messagesEl = document.getElementById('tc-messages');
    const messageEl = document.createElement('div');
    messageEl.className = 'tc-message';
    
    const time = new Date(message.timestamp).toLocaleTimeString([], { 
      hour: '2-digit', 
      minute: '2-digit' 
    });
    
    const color = getWalletColor(message.wallet);
    const shortWallet = shortenWallet(message.wallet);
    const isMe = message.wallet === state.wallet;
    
    messageEl.innerHTML = `
      <span class="tc-message-time">[${time}]</span>
      <span class="tc-message-wallet" style="color: ${color}" title="${message.wallet}">${isMe ? 'you' : shortWallet}</span>
      <span class="tc-message-text">: ${escapeHtml(message.text)}</span>
    `;
    
    // Click to copy full wallet address
    messageEl.querySelector('.tc-message-wallet').addEventListener('click', () => {
      navigator.clipboard.writeText(message.wallet);
      addSystemMessage('Copied wallet to clipboard');
    });
    
    messagesEl.appendChild(messageEl);
    messagesEl.scrollTop = messagesEl.scrollHeight;
  }

  function addSystemMessage(text) {
    const messagesEl = document.getElementById('tc-messages');
    const messageEl = document.createElement('div');
    messageEl.className = 'tc-message tc-system-message';
    messageEl.textContent = '> ' + text;
    messagesEl.appendChild(messageEl);
    messagesEl.scrollTop = messagesEl.scrollHeight;
  }

  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // ============================================
  // INITIALIZE
  // ============================================
  function init() {
    // Wait for page to be ready
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', createWidget);
    } else {
      createWidget();
    }
  }

  init();
})();
